<?php
/*
Plugin Name: PDee Internships
Description: Adds an "Internships" custom post type with a shortcode to display listings on the front‑end.
Version: 1.0.0
Author: PDee
License: GPLv2 or later
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class PDee_Internships {

    const CPT = 'pdee_internship';
    const SHORTCODE = 'pdee_internships';

    public function __construct() {
        add_action( 'init', [ $this, 'register_post_type' ] );
        add_action( 'add_meta_boxes', [ $this, 'add_meta_boxes' ] );
        add_action( 'save_post', [ $this, 'save_meta_boxes' ] );
        add_shortcode( self::SHORTCODE, [ $this, 'render_shortcode' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_styles' ] );
        register_activation_hook( __FILE__, [ __CLASS__, 'activation' ] );
        register_deactivation_hook( __FILE__, [ __CLASS__, 'deactivation' ] );
    }

    /**
     * Register the custom post type.
     */
    public function register_post_type() {
        $labels = [
            'name'               => __( 'Internships', 'pdee-internships' ),
            'singular_name'      => __( 'Internship', 'pdee-internships' ),
            'add_new'            => __( 'Add New', 'pdee-internships' ),
            'add_new_item'       => __( 'Add New Internship', 'pdee-internships' ),
            'edit_item'          => __( 'Edit Internship', 'pdee-internships' ),
            'new_item'           => __( 'New Internship', 'pdee-internships' ),
            'view_item'          => __( 'View Internship', 'pdee-internships' ),
            'search_items'       => __( 'Search Internships', 'pdee-internships' ),
            'not_found'          => __( 'No internships found', 'pdee-internships' ),
            'not_found_in_trash' => __( 'No internships found in Trash', 'pdee-internships' ),
            'menu_name'          => __( 'Internships', 'pdee-internships' ),
        ];

        $args = [
            'labels'             => $labels,
            'public'             => true,
            'has_archive'        => true,
            'show_in_rest'       => true,
            'menu_position'      => 20,
            'menu_icon'          => 'dashicons-welcome-learn-more',
            'supports'           => [ 'title', 'editor', 'thumbnail' ],
        ];

        register_post_type( self::CPT, $args );
    }

    /**
     * Add meta boxes for short description and application URL.
     */
    public function add_meta_boxes() {
        add_meta_box( 'pdee_short_desc', __( 'Short Description', 'pdee-internships' ), [ $this, 'short_desc_box' ], self::CPT, 'normal', 'high' );
        add_meta_box( 'pdee_app_url', __( 'Application URL', 'pdee-internships' ), [ $this, 'app_url_box' ], self::CPT, 'normal', 'default' );
    }

    /**
     * Short description box callback.
     */
    public function short_desc_box( $post ) {
        $value = get_post_meta( $post->ID, '_pdee_short_desc', true );
        wp_nonce_field( 'pdee_save_meta', 'pdee_meta_nonce' );
        echo '<textarea style="width:100%;min-height:80px;" id="pdee_short_desc" name="pdee_short_desc">' . esc_textarea( $value ) . '</textarea>';
    }

    /**
     * Application URL box callback.
     */
    public function app_url_box( $post ) {
        $value = get_post_meta( $post->ID, '_pdee_app_url', true );
        echo '<input type="url" style="width:100%;" id="pdee_app_url" name="pdee_app_url" value="' . esc_attr( $value ) . '" placeholder="https://example.com/apply" />';
    }

    /**
     * Save meta box data.
     */
    public function save_meta_boxes( $post_id ) {
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        if ( ! isset( $_POST['pdee_meta_nonce'] ) || ! wp_verify_nonce( $_POST['pdee_meta_nonce'], 'pdee_save_meta' ) ) {
            return;
        }

        if ( isset( $_POST['pdee_short_desc'] ) ) {
            update_post_meta( $post_id, '_pdee_short_desc', sanitize_textarea_field( $_POST['pdee_short_desc'] ) );
        }

        if ( isset( $_POST['pdee_app_url'] ) ) {
            update_post_meta( $post_id, '_pdee_app_url', esc_url_raw( $_POST['pdee_app_url'] ) );
        }
    }

    /**
     * Shortcode renderer.
     *
     * Usage: [pdee_internships count="5" order="DESC"]
     */
    public function render_shortcode( $atts ) {
        $atts = shortcode_atts( [
            'count' => -1,
            'order' => 'DESC',
        ], $atts, self::SHORTCODE );

        $query = new WP_Query( [
            'post_type'      => self::CPT,
            'posts_per_page' => intval( $atts['count'] ),
            'order'          => $atts['order'],
        ] );

        if ( ! $query->have_posts() ) {
            return '<p>' . __( 'No internships available at the moment.', 'pdee-internships' ) . '</p>';
        }

        ob_start();
        echo '<div class="pdee-internships-wrapper">';
        while ( $query->have_posts() ) {
            $query->the_post();
            $app_url       = esc_url( get_post_meta( get_the_ID(), '_pdee_app_url', true ) );
            $short_desc    = esc_html( get_post_meta( get_the_ID(), '_pdee_short_desc', true ) );
            ?>
            <article class="pdee-internship">
                <h2 class="pdee-title" style="font-weight:bold;"><?php the_title(); ?></h2>
                <?php if ( $short_desc ) : ?>
                    <p class="pdee-short-desc"><?php echo $short_desc; ?></p>
                <?php endif; ?>
                <div class="pdee-description">
                    <?php the_content(); ?>
                </div>
                <p class="pdee-date" style="font-size:0.9em;color:#777;"><?php echo esc_html( get_the_date() ); ?></p>
                <?php if ( $app_url ) : ?>
                    <p><a class="pdee-apply" href="<?php echo $app_url; ?>" target="_blank" rel="noopener noreferrer"><?php _e( 'Apply here', 'pdee-internships' ); ?></a></p>
                <?php endif; ?>
            </article>
            <?php
        }
        echo '</div>'; // .pdee-internships-wrapper
        wp_reset_postdata();
        return ob_get_clean();
    }

    /**
     * Enqueue front‑end styles.
     */
    public function enqueue_styles() {
        wp_register_style( 'pdee-internships', plugin_dir_url( __FILE__ ) . 'pdee-internships.css', [], '1.0.0' );
        wp_enqueue_style( 'pdee-internships' );
    }

    /**
     * Plugin activation: flush rewrite rules.
     */
    public static function activation() {
        $obj = new self();
        $obj->register_post_type();
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation: flush rewrite rules.
     */
    public static function deactivation() {
        flush_rewrite_rules();
    }
}

new PDee_Internships();

/* --------------------------------------------------
 * Optional basic stylesheet (pdee-internships.css)
 * --------------------------------------------------
 * .pdee-internship { margin-bottom: 2rem; }
 * .pdee-title { font-size: 1.25rem; }
 * .pdee-description { margin: 0.5rem 0 1rem 0; }
 * .pdee-date { font-style: italic; }
 * .pdee-apply { color: #0073aa; text-decoration: underline; }
 * .pdee-apply:hover { color: #005177; }
 */
